<?php
require_once '../db_connect.php'; // Include database connection

// Prevent re-seeding if data already exists
$stmt = $conn->query("SELECT COUNT(*) FROM users");
try {
    $conn->beginTransaction();

    // Sample data arrays
    $firstNames = ['John', 'Emma', 'Michael', 'Sophia', 'James', 'Olivia', 'William', 'Ava', 'Liam', 'Charlotte'];
    $lastNames = ['Smith', 'Johnson', 'Brown', 'Taylor', 'Wilson', 'Davis', 'Clark', 'Harris', 'Lewis', 'Walker'];
    $countries = ['USA', 'UK', 'Canada', 'Australia', 'Germany', 'France', 'India', 'Brazil', 'Japan', 'Nigeria'];
    $cities = ['New York', 'London', 'Toronto', 'Sydney', 'Berlin', 'Paris', 'Mumbai', 'Sao Paulo', 'Tokyo', 'Lagos'];
    $skills = ['PHP', 'JavaScript', 'Python', 'Graphic Design', 'SEO', 'Content Writing', 'Mobile App Development', 'Web Design', 'Data Analysis', 'Video Editing'];
    $jobTitles = [
        'Develop a WordPress Website', 'Create a Mobile App', 'Design a Logo', 'Write SEO Content', 'Build an E-commerce Platform',
        'Create a Marketing Video', 'Develop a Python Script', 'Design a UI/UX Mockup', 'Optimize Website SEO', 'Create a Blog'
    ];
    $experienceLevels = ['beginner', 'intermediate', 'expert'];

    // Seed Users (50 total: 30 clients, 15 freelancers, 5 admins)
    $users = [];
    for ($i = 0; $i < 50; $i++) {
        $firstName = $firstNames[array_rand($firstNames)];
        $lastName = $lastNames[array_rand($lastNames)];
        $username = strtolower($firstName . $lastName . rand(100, 999));
        $email = strtolower($firstName . '.' . $lastName . rand(100, 999) . '@example.com');
        $password = password_hash('Password123!', PASSWORD_DEFAULT);
        $role = $i < 30 ? 'client' : ($i < 45 ? 'freelancer' : 'admin');
        $fullName = "$firstName $lastName";
        $phone = '+1' . rand(1000000000, 9999999999);
        $country = $countries[array_rand($countries)];
        $city = $cities[array_rand($cities)];
        $address = rand(100, 9999) . ' Main St, ' . $city;
        $bio = "I'm a $role based in $city, specializing in " . ($role === 'freelancer' ? $skills[array_rand($skills)] : 'hiring talent') . ".";
        $rating = $role === 'freelancer' ? rand(30, 50) / 10 : 0.00;
        $totalReviews = $role === 'freelancer' ? rand(0, 20) : 0;
        $balance = rand(0, 5000) / 100;
        $isVerified = rand(0, 1);
        $active = 1;
        $lastSeen = date('Y-m-d H:i:s', strtotime('-' . rand(0, 30) . ' days'));

        $stmt = $conn->prepare("
            INSERT INTO users (username, email, password, role, full_name, phone, country, city, address, bio, rating, total_reviews, balance, is_verified, active, last_seen)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([$username, $email, $password, $role, $fullName, $phone, $country, $city, $address, $bio, $rating, $totalReviews, $balance, $isVerified, $active, $lastSeen]);
        $users[] = ['id' => $conn->lastInsertId(), 'role' => $role, 'username' => $username];
    }

    // Seed Freelancer Profiles (for freelancers)
    $freelancerIds = array_filter($users, fn($u) => $u['role'] === 'freelancer');
    foreach ($freelancerIds as $user) {
        $title = 'Professional ' . $skills[array_rand($skills)] . ' Services';
        $hourlyRate = rand(1500, 10000) / 100;
        $freelancerSkills = array_slice($skills, rand(0, count($skills) - 3), rand(1, 3));
        $experienceLevel = $experienceLevels[array_rand($experienceLevels)];

        $stmt = $conn->prepare("
            INSERT INTO freelancer_profiles (user_id, title, hourly_rate, skills, experience_level)
            VALUES (?, ?, ?, ?, ?)
        ");
        $stmt->execute([$user['id'], $title, $hourlyRate, implode(',', $freelancerSkills), $experienceLevel]);
    }

    // Seed Jobs (100 jobs by clients)
    $clientIds = array_filter($users, fn($u) => $u['role'] === 'client');
    $jobs = [];
    for ($i = 0; $i < 100; $i++) {
        $client = $clientIds[array_rand($clientIds)];
        $title = $jobTitles[array_rand($jobTitles)];
        $description = "Need a skilled professional to $title. Must have experience in " . $skills[array_rand($skills)] . ". Deadline: " . date('Y-m-d', strtotime('+30 days'));
        $budget = rand(5000, 50000) / 100;
        $status = ['open', 'in_progress', 'completed', 'cancelled'][rand(0, 3)];
        $isApproved = ['pending', 'approved', 'rejected'][rand(0, 2)];

        $stmt = $conn->prepare("
            INSERT INTO jobs (client_id, title, description, budget, status, is_approved)
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([$client['id'], $title, $description, $budget, $status, $isApproved]);
        $jobId = $conn->lastInsertId();
        $jobs[] = ['id' => $jobId, 'client_id' => $client['id']];

        // Seed Job Skills (1-3 skills per job)
        $jobSkills = array_slice($skills, rand(0, count($skills) - 3), rand(1, 3));
        foreach ($jobSkills as $skill) {
            $stmt = $conn->prepare("INSERT INTO job_skills (job_id, skill) VALUES (?, ?)");
            $stmt->execute([$jobId, $skill]);
        }
    }

    // Seed Proposals (2-5 proposals per job)
    $proposals = [];
    foreach ($jobs as $job) {
        $numProposals = rand(2, 5);
        for ($i = 0; $i < $numProposals; $i++) {
            $freelancer = $freelancerIds[array_rand($freelancerIds)];
            $coverLetter = "I'm a skilled freelancer experienced in " . $skills[array_rand($skills)] . ". I can complete this job efficiently.";
            $proposedAmount = rand(4000, 60000) / 100;
            $estimatedDays = rand(5, 30);
            $status = ['pending', 'accepted', 'rejected'][rand(0, 2)];

            $stmt = $conn->prepare("
                INSERT INTO proposals (job_id, freelancer_id, cover_letter, proposed_amount, estimated_days, status)
                VALUES (?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([$job['id'], $freelancer['id'], $coverLetter, $proposedAmount, $estimatedDays, $status]);
            if ($status === 'accepted') {
                $proposals[] = [
                    'id' => $conn->lastInsertId(),
                    'job_id' => $job['id'],
                    'freelancer_id' => $freelancer['id'],
                    'client_id' => $job['client_id']
                ];
            }
        }
    }

    // Seed Orders (for accepted proposals)
    $orders = [];
    foreach ($proposals as $proposal) {
        $status = ['in_progress', 'completed', 'cancelled'][rand(0, 2)];
        $deliveryMessage = $status === 'completed' ? 'Here is the final delivery for the job.' : null;
        $deliveryFile = $status === 'completed' ? '/uploads/sample_delivery_' . rand(1, 100) . '.zip' : null;
        $deliveredAt = $status === 'completed' ? date('Y-m-d H:i:s', strtotime('-' . rand(1, 10) . ' days')) : null;

        $stmt = $conn->prepare("
            INSERT INTO orders (job_id, proposal_id, client_id, freelancer_id, delivery_message, delivery_file, delivered_at, status)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $proposal['job_id'],
            $proposal['id'],
            $proposal['client_id'],
            $proposal['freelancer_id'],
            $deliveryMessage,
            $deliveryFile,
            $deliveredAt,
            $status
        ]);
        $orders[] = [
            'id' => $conn->lastInsertId(),
            'proposal_id' => $proposal['id'],
            'client_id' => $proposal['client_id'],
            'freelancer_id' => $proposal['freelancer_id'],
            'job_id' => $proposal['job_id'],
            'status' => $status
        ];
    }

    // Seed Contracts (for accepted proposals)
    $contracts = [];
    foreach ($proposals as $proposal) {
        $agreedAmount = rand(4000, 60000) / 100;
        $status = ['active', 'completed', 'cancelled'][rand(0, 2)];
        $endDate = $status === 'completed' ? date('Y-m-d H:i:s', strtotime('-' . rand(1, 5) . ' days')) : null;

        $stmt = $conn->prepare("
            INSERT INTO contracts (proposal_id, client_id, freelancer_id, agreed_amount, status, end_date)
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([$proposal['id'], $proposal['client_id'], $proposal['freelancer_id'], $agreedAmount, $status, $endDate]);
        $contracts[] = [
            'id' => $conn->lastInsertId(),
            'proposal_id' => $proposal['id'],
            'status' => $status
        ];
    }

    // Seed Payments (for contracts)
    foreach ($contracts as $contract) {
        $transactionId = 'TXN' . rand(100000, 999999);
        $amount = rand(4000, 60000) / 100;
        $method = ['paypal', 'stripe', 'bank'][rand(0, 2)];
        $status = ['pending', 'completed', 'failed'][rand(0, 2)];

        $stmt = $conn->prepare("
            INSERT INTO payments (contract_id, transaction_id, amount, method, status)
            VALUES (?, ?, ?, ?, ?)
        ");
        $stmt->execute([$contract['id'], $transactionId, $amount, $method, $status]);
    }

    // Seed Reviews (for completed contracts)
    foreach ($contracts as $contract) {
        if ($contract['status'] === 'completed') {
            $reviewer = $clientIds[array_rand($clientIds)];
            $reviewee = $freelancerIds[array_rand($freelancerIds)];
            $rating = rand(30, 50) / 10;
            $comment = "Great work on the project! " . ($rating >= 4 ? 'Highly recommended.' : 'Some areas could improve.');

            $stmt = $conn->prepare("
                INSERT INTO reviews (contract_id, reviewer_id, reviewee_id, rating, comment)
                VALUES (?, ?, ?, ?, ?)
            ");
            $stmt->execute([$contract['id'], $reviewer['id'], $reviewee['id'], $rating, $comment]);
        }
    }

    // Seed Wallets (for all users)
    foreach ($users as $user) {
        $balance = rand(0, 10000) / 100;
        $stmt = $conn->prepare("INSERT INTO wallets (user_id, balance) VALUES (?, ?)");
        $stmt->execute([$user['id'], $balance]);
    }

    // Seed Transactions (for users with orders)
    foreach ($orders as $order) {
        $amount = rand(1000, 5000) / 100;
        $type = ['credit', 'debit'][rand(0, 1)];
        $description = $type === 'credit' ? 'Payment received for order #' . $order['id'] : 'Withdrawal from wallet';

        $stmt = $conn->prepare("
            INSERT INTO transactions (user_id, type, amount, description)
            VALUES (?, ?, ?, ?)
        ");
        $stmt->execute([$order['freelancer_id'], $type, $amount, $description]);
    }

    // Seed Messages (for orders)
    foreach ($orders as $order) {
        $numMessages = rand(1, 5);
        for ($i = 0; $i < $numMessages; $i++) {
            $sender = rand(0, 1) ? $order['client_id'] : $order['freelancer_id'];
            $receiver = $sender === $order['client_id'] ? $order['freelancer_id'] : $order['client_id'];
            $message = "Discussion about order #" . $order['id'] . ": " . ($i % 2 ? 'Please provide an update.' : 'Working on it, will deliver soon.');
            $isRead = rand(0, 1);

            $stmt = $conn->prepare("
                INSERT INTO messages (job_id, sender_id, receiver_id, message, is_read)
                VALUES (?, ?, ?, ?, ?)
            ");
            $stmt->execute([$order['job_id'], $sender, $receiver, $message, $isRead]);
        }
    }
// Seed Disputes (1-3 disputes for some completed orders)
$completedOrders = array_filter($orders, fn($o) => $o['status'] === 'completed');
foreach ($completedOrders as $index => $order) {
    if ($index % 2 === 0) { // Seed disputes for half of completed orders
        $numDisputes = rand(1, 3);
        for ($i = 0; $i < $numDisputes; $i++) {
            $reason = "Dispute over order #" . $order['id'] . ": " . ($i % 2 ? 'Incomplete delivery' : 'Quality not as expected');
            $status = ['open', 'resolved', 'closed'][rand(0, 2)];
            $resolvedAt = $status === 'resolved' || $status === 'closed' ? date('Y-m-d H:i:s', strtotime('-' . rand(1, 5) . ' days')) : null;
            $resolution = $status === 'resolved' || $status === 'closed' ? ($i % 2 ? 'Refund issued' : 'Revised delivery accepted') : null;

            $stmt = $conn->prepare("
                INSERT INTO disputes (order_id, client_id, freelancer_id, reason, status, resolved_at, resolution)
                VALUES (?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([$order['id'], $order['client_id'], $order['freelancer_id'], $reason, $status, $resolvedAt, $resolution]);
        }
    }
}
    // Seed Notifications (for users)
    foreach ($users as $user) {
        $numNotifications = rand(1, 3);
        for ($i = 0; $i < $numNotifications; $i++) {
            $type = ['job_update', 'order_update', 'payment'][rand(0, 2)];
            $message = "Your $type has been updated: " . ($type === 'job_update' ? 'Job posted' : ($type === 'order_update' ? 'Order completed' : 'Payment processed'));
            $isRead = rand(0, 1);

            $stmt = $conn->prepare("
                INSERT INTO notifications (user_id, type, message, is_read)
                VALUES (?, ?, ?, ?)
            ");
            $stmt->execute([$user['id'], $type, $message, $isRead]);
        }
    }

    $conn->commit();
    echo "Database seeded successfully with demo data!";
} catch (PDOException $e) {
    $conn->rollBack();
    echo "Error seeding database: " . $e->getMessage();
}
?>